/**************************************************************************
 * OS/2 Sample Print Application PRTSAMP
 *
 *  File name  :  prtmenu.c
 *
 *  Description:  This file contains functions which respond to selections
 *		  the user makes at the menu bar or the pulldown menus on it.
 *
 *		  This source file contains the following functions:
 *
 *		  Menu(hwnd, msg, mp1, mp2)
 *		  GetEAFileType(hfile)
 *		  FileTypeDlgProc(hwnd, msg, mp1, mp2)
 *		  MyFileDlgProc(hwnd, msg, mp1, mp2)
 *		  MyFontDlgProc(hwnd, msg, mp1, mp2)
 *		  ProdInfoDlgProc(hwnd, msg, mp1, mp2)
 *		  SetSysMenu(hdlg)
 *
 *  Concepts   :  standard dialogs
 *
 *  API's      :  WinQueryWindowULong
 *		  WinFileDlg
 *		  WinFontDlg
 *		  WinDlgBox
 *		  WinSendMsg
 *		  WinPostMsg
 *		  WinCheckMenuItem
 *		  GpiSetDefaultViewMatrix
 *		  WinAlarm
 *		  DosQueryFileInfo
 *		  WinCheckButton
 *		  WinSetFocus
 *		  WinQueryButtonCheckstate
 *		  WinDismissDlg
 *		  DosOpen
 *		  DosRead
 *		  DosClose
 *		  WinLoadString
 *		  WinMessageBox
 *		  WinDefFileDlgProc
 *		  WinDefFontDlgProc
 *		  WinDefDlgProc
 *		  WinEnableMenuItem
 *		  WinWindowFromID
 *
 *    Files    :  OS2.H, PRTSAMP.H, PRTSDLG.H, PRTSHLP.H, PMASSERT.H
 *
 *  Copyright (C) 1991 IBM Corporation
 *
 *	DISCLAIMER OF WARRANTIES.  The following [enclosed] code is
 *	sample code created by IBM Corporation. This sample code is not
 *	part of any standard or IBM product and is provided to you solely
 *	for  the purpose of assisting you in the development of your
 *	applications.  The code is provided "AS IS", without
 *	warranty of any kind.  IBM shall not be liable for any damages
 *	arising out of your use of the sample code, even if they have been
 *	advised of the possibility of such damages.
 *************************************************************************/

/* os2 includes */
#define INCL_DOSFILEMGR
#define INCL_WINSTDFILE
#define INCL_WINSTDFONT
#define INCL_WININPUT
#define INCL_WINFRAMEMGR
#define INCL_WINWINDOWMGR
#define INCL_WINMENUS
#define INCL_WINHELP
#define INCL_WINBUTTONS
#define INCL_WINDIALOGS
#define INCL_WINERRORS
#define INCL_GPITRANSFORMS
#define INCL_SPL
#define INCL_SPLDOSPRINT
#include <os2.h>

/* c language includes */
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <stddef.h>
#include <process.h>
#include <sys\types.h>
#include <sys\stat.h>

/* application includes */
#include "prtsamp.h"
#include "prtsdlg.h"
#include "prtshlp.h"
#include "pmassert.h"

/**************************************************************************
 *
 *  Name       : Menu()
 *
 *  Description: Processes commands initiated from the menu bar.
 *
 *  Concepts:	 pull-down menus
 *
 *  API's      :  WinQueryWindowULong
 *		  WinFileDlg
 *		  WinFontDlg
 *		  WinDlgBox
 *		  WinSendMsg
 *		  WinPostMsg
 *		  WinCheckMenuItem
 *		  GpiSetDefaultViewMatrix
 *		  WinAlarm
 *
 *  Parameters :  hwnd = window handle
 *		  msg  = message code
 *		  mp1  = first message parameter
 *		  mp2  = second message parameter
 *
 *  Return     :  depends on message sent
 *************************************************************************/
MRESULT Menu( HWND hwnd, ULONG msg, MPARAM mp1, MPARAM mp2 )
{
   PMAIN_PARM pmp;
   ULONG      rc;
   BOOL	      bOK;
   POINTL     pointl;
   FIXED      aFixed[ 2 ];
   FIXED      fxWork;
   LONG	      lRC;

   /* obtain the main parameter pointer from window words */
   pmp = (PMAIN_PARM)WinQueryWindowULong( hwnd, QWL_USER );

   switch( msg )
   {
   case WM_COMMAND:
      switch(SHORT1FROMMP(mp1))
      {
      case IDM_OPEN:
	/* filedlg structure initialized in create.c.
	 * See function MyFileDlgProc() which operates while file dialog is up.
	 */
	 WinFileDlg( HWND_DESKTOP, hwnd, &pmp->filedlg );
	 return (MRESULT)NULL;

      case IDM_PAGEDOWN:
	/* preview the next page of text */
	 WinSendMsg( hwnd, WM_USER_DISABLE_CLIENT, (MPARAM)0L, (MPARAM)0L );
	 WinPostMsg( pmp->hwndObject, WM_USER_PAGINATE,
			       (MPARAM)hwnd, (MPARAM)FLAGS_PAGINATE_SCREEN );
	 return (MRESULT)NULL;

      case IDM_PRINT:
	 Print(hwnd, pmp);
	 return (MRESULT)NULL;

      case IDM_HELPINDEX:
	 rc = (ULONG)WinSendMsg(pmp->hwndHelp, HM_HELP_INDEX,
			     (MPARAM)NULL, (MPARAM)NULL);
	 pmassert( pmp->hab, rc == 0L );
	 break;

      case IDM_HELPEXTENDED:
	 rc = (ULONG)WinSendMsg(pmp->hwndHelp, HM_EXT_HELP,
			     (MPARAM)NULL, (MPARAM)NULL);
	 pmassert( pmp->hab, rc == 0L );
	 break;

      case IDM_USINGHELP:
	 rc = (ULONG)WinSendMsg(pmp->hwndHelp, HM_DISPLAY_HELP,
			      (MPARAM)NULL, (MPARAM)NULL);
	 pmassert( pmp->hab, rc == 0L );
	 break;

      case IDM_PRODINFO:
	 rc = WinDlgBox( HWND_DESKTOP,
			hwnd,
			(PFNWP)ProdInfoDlgProc,
			(HMODULE)NULLHANDLE,
			IDD_PRODINFO,
			(PVOID)pmp );
	 pmassert( pmp->hab, rc != DID_ERROR );
	 return (MRESULT)NULL;

      case IDM_SETFONT:
	/* let user pick a font */
	pmassert( pmp->hab, pmp->ulMode == MODE_TEXT );

	/* reassign hpsPrinter because it could have changed */
	pmp->fontdlg.hpsPrinter	  = pmp->hpsPrinterInfo;


	WinFontDlg( HWND_DESKTOP, hwnd, &pmp->fontdlg );
	if( pmp->fontdlg.lReturn == DID_OK )
	{
	   /*
	    * Store an indicator that the font dialog was recently visited.
	    * This indicator is referenced in PRTSAMP.C under the
	    * WM_NACK_DEFAULT_FONT case where the user is warned that
	    * the pagination routine is using a default font.
	    */
	   pmp->fVisitedFontDialog = TRUE;

	   /* requested vector (outline) fonts only */
	   pmassert( pmp->hab, pmp->fontdlg.fAttrs.fsFontUse & FATTR_FONTUSE_OUTLINE )

	   if( pmp->f )
	   {
	      /* seek top of file to display document in new font */
	      fseek( pmp->f, 0, SEEK_SET );
	      /* re-paginate for the screen */
	      WinSendMsg( hwnd, WM_USER_DISABLE_CLIENT, 0, 0 );
	      WinPostMsg( pmp->hwndObject, WM_USER_PAGINATE,
				   (MPARAM)hwnd, (MPARAM)FLAGS_PAGINATE_SCREEN );
	   }
	}

	 return (MRESULT)NULL;


      case IDM_SETUP:
	/*
	 * Present queue selection dialog to user (parm 3 == TRUE);
	 * Returns TRUE if queue/printer/form options changed.
	 * If changed, then reset the program with NEW_MODE processing. See
	 * prtsamp.c for the WM_USER_NEW_MODE case.
	 */
	if( QueryPrintQueue( pmp, TRUE ))
	{
	    WinPostMsg(pmp->hwndClient, WM_USER_NEW_MODE, 0, 0);
	}
	 return (MRESULT)NULL;


      case IDM_VIEWHALF:
      case IDM_VIEWFULL:
	 switch ( SHORT1FROMMP (mp1) )
	 {
	 case IDM_VIEWHALF:
	    WinCheckMenuItem( pmp->hwndMenubar, IDM_VIEWFULL, FALSE );
	    WinCheckMenuItem( pmp->hwndMenubar, IDM_VIEWHALF, TRUE  );
	    fxWork = MAKEFIXED( 0, 32768 );
	    pmp->floatScale = 0.50;
	    break;

	case IDM_VIEWFULL:
	    WinCheckMenuItem( pmp->hwndMenubar, IDM_VIEWFULL, TRUE  );
	    WinCheckMenuItem( pmp->hwndMenubar, IDM_VIEWHALF, FALSE );
	    fxWork = MAKEFIXED( 1, 0 );
	    pmp->floatScale = 1.00;
	    break;
	}

	if( pmp->matlfDefView.fxM11 != fxWork )
	{
	   /* apply the new scaling with no Y translation for now */
	    pmp->matlfDefView.fxM11 = fxWork;
	    pmp->matlfDefView.fxM22 = fxWork;
	    pmp->matlfDefView.lM31  = OFFSET_XY_TWIPS;
	    pmp->matlfDefView.lM32  = 0;
	    bOK = GpiSetDefaultViewMatrix( pmp->hpsClient, 9,
				   &pmp->matlfDefView, TRANSFORM_REPLACE );
	    pmassert( pmp->hab, bOK );

	   /*
	    * Still need Y translation to "home" the document in the client
	    * window. "Home" means to position the document such that the
	    * top-left corner of the document appears in the top-left corner
	    * of the client window.
	    *
	    * The WM_SIZE case in prtsamp.c sets a proper Y translation so it
	    * homes the document
	    *
	    */
	    WinSendMsg( hwnd, WM_SIZE, (MPARAM)0L, (MPARAM)0L );
	 }
	 else
	 {
	   /* viewmatrix ok already */
	    WinAlarm( HWND_DESKTOP, WA_NOTE );
	 }
	 return (MRESULT)NULL;
      }
      break;
   }
   return (MRESULT)NULL;
}   /*	end of Menu()  */


/**************************************************************************
 *
 *  Name       : GetEAFileType
 *
 *  Description: Gets the value for the extended attribute called ".TYPE".
 *		 For more information, see the EA Sample Program and
 *		 BSEDOS.H.
 *
 *  Concepts:	 extended attributes
 *
 *  API's      : DosQueryFileInfo
 *
 *  Parameters :  hfile = file handle to query
 *
 *  Return     :  the extended attribute file type MODE_* as a ULONG
 *************************************************************************/
ULONG GetEAFileType( HFILE hfile )
{
   ULONG       ulMode;
   ULONG       rc;
   EAOP2       eaop2;
   PGEA2LIST   pgea2list;
   PFEA2LIST   pfea2list;
   PGEA2       pgea2;
   PFEA2       pfea2;
   PUSHORT     pus;

   /* default to unknown */
   ulMode = MODE_UNKNOWN;

   /* create one list item in GEA list */
   pgea2 = (PGEA2) calloc ( 1, 10 );
   pgea2->oNextEntryOffset = 0;
   pgea2->cbName	   = 5;
   memcpy( pgea2->szName, ".TYPE", 5 );

   /* create the list with one GEA item */
   pgea2list = (PGEA2LIST) calloc ( 1, 14 );
   pgea2list->cbList = 10;
   memcpy( &pgea2list->list, pgea2, 10 );

   /* create the list where DosQueryFileInfo places result */
   pfea2list = (PFEA2LIST) calloc ( 1, 64 );
   pfea2list->cbList = 64;

   /* put both lists in a eaop2 structure */
   eaop2.fpGEA2List = pgea2list;
   eaop2.fpFEA2List = pfea2list;

   /* get info */
   if( 0 == DosQueryFileInfo( hfile, FIL_QUERYEASFROMLIST, &eaop2, 64 ) )
   {
     /* is there a ".TYPE" EA? */
      pfea2 = pfea2list->list;
      if( pfea2->cbValue > 0 )
      {
       /* Point to the EAT_* value. See bsedos.h or the EA sample program. */
	 pus = (PUSHORT) (PCHAR)(((PCHAR)pfea2)+20);
	 switch( *pus )
	 {
	 case EAT_ASCII:
	    ulMode = MODE_TEXT;
	    break;
	 case EAT_BITMAP:
	    ulMode = MODE_BITMAP;
	    break;
	 case EAT_METAFILE:
	    ulMode = MODE_METAFILE;
	    break;
	 }
      }
   }
   free( pfea2list );
   free( pgea2list );
   free( pgea2 );
   return ulMode;
}     /*  end of GetEAFileType()  */


/**************************************************************************
 *
 *  Name       : FileTypeDlgProc
 *
 *  Description: Window procedure for the dialog that lets the user
 *		 indicate the file type, presented as a child dialog
 *		 of the standard file dialog.
 *
 *  Concepts:	 window words
 *
 *  API's      : WinSetWindowULong
 *		 WinCheckButton
 *		 WinSetFocus
 *		 WinQueryButtonCheckstate
 *		 WinDismissDlg
 *		 WinDefDlgProc
 *
 *  Parameters : sets check according to pmp->ulNextMode, defaults to
 *		 text type
 *		 pmp is passed to this dialog at WM_INITDLG time and stored
 *		 in dialog window words.
 *
 *  Return     : sets pmp->ulNextMode according to user's selection
 *************************************************************************/
ULONG APIENTRY FileTypeDlgProc( HWND hwnd, ULONG msg, MPARAM mp1, MPARAM mp2 )
{
   PMAIN_PARM  pmp;
   ULONG       id;

   switch( msg )
   {
   case WM_INITDLG:
      pmp = (PMAIN_PARM) mp2;
      WinSetWindowULong( hwnd, QWL_USER, (ULONG)mp2 );

    /* set check according to pmp->ulNextMode */
      switch( pmp->ulNextMode )
      {
      case MODE_BITMAP:
	 id = IDC_CK_BITMAP;
	 break;
      case MODE_METAFILE:
	 id = IDC_CK_METAFILE;
	 break;
      default:
	 id = IDC_CK_TEXT;
	 break;
      }

    /* pmwin.h macro */
   WinCheckButton( hwnd, id, TRUE );

   WinSetFocus( HWND_DESKTOP, hwnd );

   return 1L;

   case WM_COMMAND:
      pmp = (PMAIN_PARM) WinQueryWindowULong( hwnd, QWL_USER );
      switch( SHORT1FROMMP( mp1 ))
      {
      case DID_OK:
      /* get checked button */
	 if( WinQueryButtonCheckstate( hwnd, IDC_CK_TEXT ))
	 {
	    pmp->ulNextMode = MODE_TEXT;
	 }
	 if( WinQueryButtonCheckstate( hwnd, IDC_CK_BITMAP ))
	 {
	    pmp->ulNextMode = MODE_BITMAP;
	 }
	 if( WinQueryButtonCheckstate( hwnd, IDC_CK_METAFILE ))
	 {
	    pmp->ulNextMode = MODE_METAFILE;
	 }
	 WinDismissDlg( hwnd, DID_OK );
	 break;

    case DID_CANCEL:
      /* return */
	WinDismissDlg( hwnd, DID_CANCEL );
	break;
      }
      return 0L;

   }
   return (ULONG)WinDefDlgProc( hwnd, msg, mp1, mp2 );
}  /*  end of FileTypeDlgProc()	 */


/************************************************************************
 *
 * Name:     MyFileDlgProc
 *
 * Description: Window procedure that subclasses the WinFileDlg window
 *		procedure, processes/validates filenames the user picked
 *		from the WinFileDlg.
 *		The address of this function has been specified in the
 *		pmp->FILEDLG structure before invoking WinFileDlg().
 *		Handle WM_HELP from font dialog.
 *
 *  Concepts:	 open/saveas file dialog
 *
 *  API's      : WinQueryWindowULong
 *		 WinDlgBox
 *		 WinPostMsg
 *		 DosOpen
 *		 DosRead
 *		 DosClose
 *		 WinLoadString
 *		 WinMessageBox
 *		 WinSendMsg
 *		 WinDefFileDlgProc
 *
 * Parameters:	Regular window procedure arguments.
 *		Can get pmp from filedlg.ulUser.
 *		Can get to filedlg structure by getting its pointer
 *		from dialog window words.
 *
 * Result:	presents the filetype dialog as a child of the filedlg
**************************************************************************/
MRESULT APIENTRY MyFileDlgProc( HWND hwnd, ULONG msg, MPARAM mp1, MPARAM mp2 )
{
   BOOL	       bOK;
   HFILE       hfile;
   ULONG       ulAction, rc, ulMode, ulPanel, ulBytes;
   PCHAR       pch;
   MRESULT     mr;
   char	       szWork[ LEN_WORKSTRING ];
   PMAIN_PARM  pmp;
   PFILEDLG    pfiledlg;

   switch( msg )
   {
   case FDM_VALIDATE:
      /* get pointer to fontdlg structure from window words of this dialog */
      pfiledlg = (PFILEDLG)WinQueryWindowULong( hwnd, QWL_USER );
      /* filedlg structure has a pmp pointer stored in the user dword */
      pmp = (PMAIN_PARM)pfiledlg->ulUser;


      /* file dialog asking me to validate the filename pointed to by mp1 */
      rc = DosOpen(  (PSZ)mp1,
		     &hfile,
		     &ulAction,
		     0,
		     FILE_NORMAL,
		     FILE_OPEN,
		     OPEN_ACCESS_READONLY | OPEN_SHARE_DENYNONE,
		     NULL );
      if( rc == 0 )
      {
	/* file exists; what kind of file is this according to the EAs? */
	ulMode = GetEAFileType( hfile );

	if( ulMode == MODE_UNKNOWN )
	{
	  /* not sure. Look at file extension to determine file type. */
	  pch = (PSZ)mp1 + strlen( (PSZ)mp1 );
	  pch = max( (PSZ)mp1, pch-4 );
	  if( 0 == stricmp( pch, ".BMP" )) ulMode = MODE_BITMAP;
	  if( 0 == stricmp( pch, ".TXT" )) ulMode = MODE_TEXT;
	  if( 0 == stricmp( pch, ".MET" )) ulMode = MODE_METAFILE;
	  if( 0 == stricmp( pch, ".SPL" )) ulMode = MODE_METAFILE;
	}

	if( ulMode == MODE_UNKNOWN )
	{
	  /* still not sure; peek at data in the file */
	  if( 0 == DosRead( hfile, szWork, 16, &ulBytes ))
	  {
	    if( 0==strncmp(szWork,"BM",2 ) || 0==strncmp(szWork,"BA", 2 ))
	    {
	       ulMode = MODE_BITMAP;
	    }
	    else if( szWork[3] == 0xA8	&&  szWork[4] == 0xA8  )
	    {
	       ulMode = MODE_METAFILE;
	    }
	  }
	}

	/* set "best-guess" mode into main params for FileTypeDlgProc() */
	pmp->ulNextMode = ulMode;

	/* confirm file type with user via the file type dialog */
	rc = WinDlgBox( HWND_DESKTOP,
			hwnd,
			(PFNWP)FileTypeDlgProc,
			(HMODULE)NULLHANDLE,
			IDD_FILETYPE,
			(PVOID)pmp );
	pmassert( pmp->hab, rc != DID_ERROR );
	if( rc == DID_OK )
	{
	   /* set up for new operating mode on new file */
	   strcpy( pmp->szNextFilename, pmp->filedlg.szFullFile );
	   WinPostMsg( pmp->hwndClient, WM_USER_NEW_MODE,
		      (MPARAM)NULL, (MPARAM)NULL);
	   mr = (MRESULT) 1L;
	}
	else
	{
	   mr = (MRESULT)NULL;
	}
	DosClose( hfile );
      }
      else
      {
	/* file not found */
	bOK = WinLoadString( pmp->hab, (HMODULE)NULLHANDLE,
			     ERRMSG_FILE_NOT_FOUND, LEN_WORKSTRING, szWork );
	pmassert( pmp->hab, bOK );
	WinMessageBox( HWND_DESKTOP, hwnd, szWork,
					      pmp->pszTitle, 0, MB_CANCEL );
	mr = (MRESULT)NULL;
      }
      return mr;


   case WM_HELP:
     /* get pointer to fontdlg structure from window words of this dialog */
      pfiledlg = (PFILEDLG) WinQueryWindowULong( hwnd, QWL_USER );
     /* filedlg structure has a pmp pointer stored in the user dword */
      pmp = (PMAIN_PARM) pfiledlg->ulUser;

      if( pmp->hwndHelp )
      {
	 ulPanel = PANEL_FILEDIALOG;
	 WinSendMsg( pmp->hwndHelp, HM_DISPLAY_HELP,
		   (MPARAM)&ulPanel, (MPARAM)HM_RESOURCEID );
	 return (MRESULT)NULL;
      }
      break;

   default:
      break;
   }
   return WinDefFileDlgProc( hwnd, msg, mp1, mp2 );
}   /*	end of MyFileDlgProc()	*/



/************************************************************************
*
* Name: MyFontDlgProc
*
* Description:	Window procedure that subclasses the WinFontDlg window
*		procedure.  Exists to handle WM_HELP message from the
*		standard font dialog.
*
* Parameters:	Regular window procedure arguments.
*		Can get pmp from fontdlg.ulUser.  Can get to fontdlg
*		structure by getting its pointer from dialog window words.
*
* returns: depends on message sent
*
**************************************************************************/
MRESULT APIENTRY MyFontDlgProc( HWND hwnd, ULONG msg, MPARAM mp1, MPARAM mp2 )
{
   PMAIN_PARM  pmp;
   PFONTDLG    pfontdlg;
   ULONG       ulPanel;

   switch( msg )
   {
   case WM_HELP:
     /* get pointer to filedlg structure from window words of this dialog */
      pfontdlg = (PFONTDLG) WinQueryWindowULong( hwnd, QWL_USER );
     /* fontdlg structure has a pmp pointer stored in the user dword */
      pmp = (PMAIN_PARM) pfontdlg->ulUser;

      if( pmp->hwndHelp )
      {
	 ulPanel = PANEL_FONTDIALOG;
	 WinSendMsg( pmp->hwndHelp, HM_DISPLAY_HELP,
		   (MPARAM)&ulPanel, (MPARAM)HM_RESOURCEID );
	 return (MRESULT)NULL;
      }
      break;
   }
   return WinDefFontDlgProc( hwnd, msg, mp1, mp2 );
}   /*	end of MyFontDlgProc()	*/


/*********************************************************************
*  Name : ProdInfoDlgProc
*
*  Description : Processes all messages sent to the Product
*		 Information dialog box.
*
*  Concepts : Called for each message sent to the Product
*	      Information dialog box.  The Product
*	      Information box only has a button control so
*	      this routine only processes WM_COMMAND
*	      messages.	 Any WM_COMMAND posted must have come
*	      from the Ok button so we dismiss the dialog
*	      upon receiving it.
*
*  API's : WinDismissDlg
*	   WinDefDlgProc
*
* Parameters   : hwnd - Window handle to which message is addressed
*		 msg - Message type
*		 mp1 - First message parameter
*		 mp2 - Second message parameter
*
*  Returns : Dependent upon message sent
****************************************************************/
MRESULT EXPENTRY ProdInfoDlgProc( HWND hwnd, ULONG msg, MPARAM mp1, MPARAM mp2)
{
   switch(msg)
   {
      case WM_INITDLG:
	 SetSysMenu(hwnd);	 /* system menu for this dialog	 */
	 return MRFROMSHORT(FALSE);

      case WM_COMMAND:
	 /*
	  * No matter what the command, close the dialog
	  */
	 WinDismissDlg(hwnd, TRUE);
	 break;

      default:
	 return(WinDefDlgProc(hwnd, msg, mp1, mp2));
	 break;
   }
   return (MRESULT)NULL;
}   /*	end of ProdInfoDlgProc()  */


/**************************************************************************
 *
 *  Name       : SetSysMenu(hDlg)
 *
 *  Description: Sets only the Move and Close items of the system menu
 *
 *  Concepts:  Any dialog box is free to call this routine, to edit
 *	       which menu items will appear on its System Menu pulldown.
 *
 *  API's      :  WinWindowFromID
 *		  WinSendMsg
 *
 *  Parameters :  hDlg	   = window handle of the dialog
 *
 *  Return     :  [none]
 *
 *************************************************************************/
VOID SetSysMenu(HWND hDlg)
{
    HWND     hSysMenu;
    MENUITEM Mi;
    ULONG    Pos;
    MRESULT  Id;
    SHORT    cItems;

    /******************************************************************/
    /*	We only want Move and Close in the system menu.		      */
    /******************************************************************/

    hSysMenu = WinWindowFromID(hDlg, FID_SYSMENU);
    WinSendMsg( hSysMenu, MM_QUERYITEM
	      , MPFROM2SHORT(SC_SYSMENU, FALSE), MPFROMP((PCH) & Mi));
    Pos = 0L;
    cItems = (SHORT)WinSendMsg( Mi.hwndSubMenu, MM_QUERYITEMCOUNT,
				(MPARAM)NULL, (MPARAM)NULL);
    while (cItems--)
    {
	Id = WinSendMsg( Mi.hwndSubMenu, MM_ITEMIDFROMPOSITION
			  , MPFROMLONG(Pos), (MPARAM)NULL);
	switch (SHORT1FROMMR(Id))
	{
	case SC_MOVE:
	case SC_CLOSE:
	    Pos++;  /* Don't delete that one. */
	    break;
	default:
	    WinSendMsg( Mi.hwndSubMenu, MM_DELETEITEM
		      , MPFROM2SHORT((USHORT)Id, TRUE), (MPARAM)NULL);
	}
    }
}   /*	End of SetSysMenu  */

/***************************  End of prtmenu.c	****************************/
